/************************************************************************
	KORG SYRO  func
 ***********************************************************************/
#include <stdlib.h>
#include <string.h>
#include "korg_syro_type.h"
#include "korg_syro_func.h"

static const uint8_t ecc_table[256] = {
    0x00,0x55,0x56,0x03,0x59,0x0C,0x0F,0x5A,0x5A,0x0F,0x0C,0x59,0x03,0x56,0x55,0x00,
    0x65,0x30,0x33,0x66,0x3C,0x69,0x6A,0x3F,0x3F,0x6A,0x69,0x3C,0x66,0x33,0x30,0x65,
    0x66,0x33,0x30,0x65,0x3F,0x6A,0x69,0x3C,0x3C,0x69,0x6A,0x3F,0x65,0x30,0x33,0x66,
    0x03,0x56,0x55,0x00,0x5A,0x0F,0x0C,0x59,0x59,0x0C,0x0F,0x5A,0x00,0x55,0x56,0x03,
    0x69,0x3C,0x3F,0x6A,0x30,0x65,0x66,0x33,0x33,0x66,0x65,0x30,0x6A,0x3F,0x3C,0x69,
    0x0C,0x59,0x5A,0x0F,0x55,0x00,0x03,0x56,0x56,0x03,0x00,0x55,0x0F,0x5A,0x59,0x0C,
    0x0F,0x5A,0x59,0x0C,0x56,0x03,0x00,0x55,0x55,0x00,0x03,0x56,0x0C,0x59,0x5A,0x0F,
    0x6A,0x3F,0x3C,0x69,0x33,0x66,0x65,0x30,0x30,0x65,0x66,0x33,0x69,0x3C,0x3F,0x6A,
    0x6A,0x3F,0x3C,0x69,0x33,0x66,0x65,0x30,0x30,0x65,0x66,0x33,0x69,0x3C,0x3F,0x6A,
    0x0F,0x5A,0x59,0x0C,0x56,0x03,0x00,0x55,0x55,0x00,0x03,0x56,0x0C,0x59,0x5A,0x0F,
    0x0C,0x59,0x5A,0x0F,0x55,0x00,0x03,0x56,0x56,0x03,0x00,0x55,0x0F,0x5A,0x59,0x0C,
    0x69,0x3C,0x3F,0x6A,0x30,0x65,0x66,0x33,0x33,0x66,0x65,0x30,0x6A,0x3F,0x3C,0x69,
    0x03,0x56,0x55,0x00,0x5A,0x0F,0x0C,0x59,0x59,0x0C,0x0F,0x5A,0x00,0x55,0x56,0x03,
    0x66,0x33,0x30,0x65,0x3F,0x6A,0x69,0x3C,0x3C,0x69,0x6A,0x3F,0x65,0x30,0x33,0x66,
    0x65,0x30,0x33,0x66,0x3C,0x69,0x6A,0x3F,0x3F,0x6A,0x69,0x3C,0x66,0x33,0x30,0x65,
    0x00,0x55,0x56,0x03,0x59,0x0C,0x0F,0x5A,0x5A,0x0F,0x0C,0x59,0x03,0x56,0x55,0x00
};

static const uint16_t crc16_table[] = {
	0x0000, 0x1021, 0x2042, 0x3063, 0x4084, 0x50a5, 0x60c6, 0x70e7, 
	0x8108, 0x9129, 0xa14a, 0xb16b, 0xc18c, 0xd1ad, 0xe1ce, 0xf1ef, 
	0x1231, 0x0210, 0x3273, 0x2252, 0x52b5, 0x4294, 0x72f7, 0x62d6, 
	0x9339, 0x8318, 0xb37b, 0xa35a, 0xd3bd, 0xc39c, 0xf3ff, 0xe3de, 
	0x2462, 0x3443, 0x0420, 0x1401, 0x64e6, 0x74c7, 0x44a4, 0x5485, 
	0xa56a, 0xb54b, 0x8528, 0x9509, 0xe5ee, 0xf5cf, 0xc5ac, 0xd58d, 
	0x3653, 0x2672, 0x1611, 0x0630, 0x76d7, 0x66f6, 0x5695, 0x46b4, 
	0xb75b, 0xa77a, 0x9719, 0x8738, 0xf7df, 0xe7fe, 0xd79d, 0xc7bc, 
	0x48c4, 0x58e5, 0x6886, 0x78a7, 0x0840, 0x1861, 0x2802, 0x3823, 
	0xc9cc, 0xd9ed, 0xe98e, 0xf9af, 0x8948, 0x9969, 0xa90a, 0xb92b, 
	0x5af5, 0x4ad4, 0x7ab7, 0x6a96, 0x1a71, 0x0a50, 0x3a33, 0x2a12, 
	0xdbfd, 0xcbdc, 0xfbbf, 0xeb9e, 0x9b79, 0x8b58, 0xbb3b, 0xab1a, 
	0x6ca6, 0x7c87, 0x4ce4, 0x5cc5, 0x2c22, 0x3c03, 0x0c60, 0x1c41, 
	0xedae, 0xfd8f, 0xcdec, 0xddcd, 0xad2a, 0xbd0b, 0x8d68, 0x9d49, 
	0x7e97, 0x6eb6, 0x5ed5, 0x4ef4, 0x3e13, 0x2e32, 0x1e51, 0x0e70, 
	0xff9f, 0xefbe, 0xdfdd, 0xcffc, 0xbf1b, 0xaf3a, 0x9f59, 0x8f78, 
	0x9188, 0x81a9, 0xb1ca, 0xa1eb, 0xd10c, 0xc12d, 0xf14e, 0xe16f, 
	0x1080, 0x00a1, 0x30c2, 0x20e3, 0x5004, 0x4025, 0x7046, 0x6067, 
	0x83b9, 0x9398, 0xa3fb, 0xb3da, 0xc33d, 0xd31c, 0xe37f, 0xf35e, 
	0x02b1, 0x1290, 0x22f3, 0x32d2, 0x4235, 0x5214, 0x6277, 0x7256, 
	0xb5ea, 0xa5cb, 0x95a8, 0x8589, 0xf56e, 0xe54f, 0xd52c, 0xc50d, 
	0x34e2, 0x24c3, 0x14a0, 0x0481, 0x7466, 0x6447, 0x5424, 0x4405, 
	0xa7db, 0xb7fa, 0x8799, 0x97b8, 0xe75f, 0xf77e, 0xc71d, 0xd73c, 
	0x26d3, 0x36f2, 0x0691, 0x16b0, 0x6657, 0x7676, 0x4615, 0x5634, 
	0xd94c, 0xc96d, 0xf90e, 0xe92f, 0x99c8, 0x89e9, 0xb98a, 0xa9ab,	
	0x5844, 0x4865, 0x7806, 0x6827, 0x18c0, 0x08e1, 0x3882, 0x28a3, 
	0xcb7d, 0xdb5c, 0xeb3f, 0xfb1e, 0x8bf9, 0x9bd8, 0xabbb, 0xbb9a, 
	0x4a75, 0x5a54, 0x6a37, 0x7a16, 0x0af1, 0x1ad0, 0x2ab3, 0x3a92, 
	0xfd2e, 0xed0f, 0xdd6c, 0xcd4d, 0xbdaa, 0xad8b, 0x9de8, 0x8dc9, 
	0x7c26, 0x6c07, 0x5c64, 0x4c45, 0x3ca2, 0x2c83, 0x1ce0, 0x0cc1, 
	0xef1f, 0xff3e, 0xcf5d, 0xdf7c, 0xaf9b, 0xbfba, 0x8fd9, 0x9ff8, 
	0x6e17, 0x7e36, 0x4e55, 0x5e74, 0x2e93, 0x3eb2, 0x0ed1, 0x1ef0 
};

static const int16_t sin_table[] = {
	0, 	23169, 	32767, 	23169, 	0, 	-23169, -32767, -23169
};

/*----------------------------------------------------------------
	Calculate CRC16
 ----------------------------------------------------------------*/
uint16_t SyroFunc_CalculateCrc16(uint8_t *pSrc, int size)
{
	int i;
	uint16_t dat, crc;

	crc = 0xffff;
	
	for (i=0; i<size; i++) {
		dat = *pSrc++;
		crc = crc16_table[(crc >> 8) ^ dat] ^ ((crc&0xff)<<8);
	}
	return crc;
}

/*----------------------------------------------------------------
	Calculate ECC
 ----------------------------------------------------------------*/
uint32_t SyroFunc_CalculateEcc(uint8_t *pSrc, int size)
{
	int i;
	uint8_t ecc_reg1, ecc_reg2, ecc_reg3;
	uint8_t ecc1, ecc2, bitpat_a, bitpat_b;
	uint8_t ci;
	uint32_t ecc;

	ecc_reg1 = 0;
	ecc_reg2 = 0;
	ecc_reg3 = 0;	

	for (i=0; i<size; i++) {
		ci = ecc_table[pSrc[i]];
		ecc_reg1 ^= ci;
		if (ci & 0x40) {
			ecc_reg3 ^= (uint8_t)i;
			ecc_reg2 ^= ~((uint8_t)i);
		}
	}

	ecc1 = 0;
	ecc2 = 0;
	bitpat_a = 0x80;

	bitpat_b = 0x80;
	for (i=0; i<4; i++) {
		if (ecc_reg3 & bitpat_a) {
			ecc1 |= bitpat_b;
		}
		bitpat_b >>= 1;
		if (ecc_reg2 & bitpat_a) {
			ecc1 |= bitpat_b;
		}
		bitpat_b >>= 1;
		bitpat_a >>= 1;
	}

	bitpat_b = 0x80;
	for (i=0; i<4; i++) {
		if (ecc_reg3 & bitpat_a) {
			ecc2 |= bitpat_b;
		}
		bitpat_b >>= 1;
		if (ecc_reg2 & bitpat_a) {
			ecc2 |= bitpat_b;
		}
		bitpat_b >>= 1;
		bitpat_a >>= 1;
	}		

	ecc_reg1 = (ecc_reg1 << 2) | 3;	
	
	ecc = ecc_reg1;
	ecc <<= 8;
	ecc |= ecc2;
	ecc <<= 8;
	ecc |= ecc1;
	
	return ecc;
}

/*-----------------------------------------------------------------------
	Set Tx Size
 -----------------------------------------------------------------------*/
void SyroFunc_SetTxSize(uint8_t *ptr, uint32_t size, int num_of_bytes)
{
	int i;
	
	for (i=0; i<num_of_bytes; i++) {
		*ptr++ = (uint8_t)size;
		size >>= 8;
	}
}

/*----------------------------------------------------------------
	Get sin value 
 ----------------------------------------------------------------*/
static int16_t SyroFunc_GetSinValue(int phase, bool bData)
{
	int32_t ret;
	
	ret = sin_table[phase];
	
	if (bData) {
		if (ret > 0) {
			ret = 32767 - ret;
			ret = ((ret * ret) / 32767);
			ret = 32767 - ret;
		} else if (ret < 0) {
			ret += 32767;
			ret = ((ret * ret) / 32767);
			ret -= 32767;
		}
	}

	return (int16_t)ret;
}

/*-----------------------------------------------------------------------
	Generate Single Sycle
 -----------------------------------------------------------------------*/
void SyroFunc_GenerateSingleCycle(SyroChannel *psc, int write_page, uint8_t dat, bool block)
{
	int i, phase_org, phase;
	int32_t dat1, dat2;
	int vol, dlt;
	int write_pos, write_pos_last;
	
	write_pos = write_page * KORGSYRO_QAM_CYCLE;
	write_pos_last = write_pos ? (write_pos - 1) : (KORGSYRO_NUM_OF_CYCLE_BUF - 1);
	
	phase_org = (dat >> 1) & 3;
	phase = phase_org * (KORGSYRO_QAM_CYCLE / 4);
	vol = (dat & 1);
	vol = vol ? 16 : 4;
	
	for (i=0; i<KORGSYRO_QAM_CYCLE; i++) {
		dat1 = SyroFunc_GetSinValue(phase, block);
		dat1 = (dat1 * vol) / 24;
		if (!i) {
			if (phase_org != psc->LastPhase) {
				if (((psc->LastPhase & 1) && (phase_org & 1)) ||
					(((psc->LastPhase + 1) & 3) == phase_org))
				{
					dat2 = psc->CycleSample[write_pos_last];
					dlt = dat1 - dat2;
					dlt /= 3;
					dat1 -= dlt;
					dat2 += dlt;
					psc->CycleSample[write_pos_last] = (int16_t)dat2;
				}
			}
		}
		
		psc->CycleSample[write_pos++] = (int16_t)dat1;
		if ((++phase) == KORGSYRO_QAM_CYCLE) {
			phase = 0;
		}
	}
	psc->LastPhase = phase_org;
}

/*-----------------------------------------------------------------------
	Smooth Start Mark
 -----------------------------------------------------------------------*/
static void SyroFunc_SmoothStartMark(SyroChannel *psc, int write_page)
{
	int write_pos, write_pos_last;
	int32_t dat1, dat2, dat3, avg;
	
	write_pos = write_page * KORGSYRO_QAM_CYCLE;
	write_pos_last = write_pos ? (write_pos - 1) : (KORGSYRO_NUM_OF_CYCLE_BUF - 1);
	
	dat1 = psc->CycleSample[write_pos_last];
	dat2 = psc->CycleSample[write_pos];
	dat3 = psc->CycleSample[write_pos+1];

	avg = (dat1 + dat2 + dat3) / 3;
	
	dat1 = (dat1 + avg) / 2;
	dat2 = (dat2 + avg) / 2;
	dat3 = (dat3 + avg) / 2;
	
	psc->CycleSample[write_pos_last] = (int16_t)dat1;
	psc->CycleSample[write_pos] = (int16_t)dat2;
	psc->CycleSample[write_pos+1] = (int16_t)dat3;
}

/*-----------------------------------------------------------------------
	Generate Gap
 -----------------------------------------------------------------------*/
void SyroFunc_MakeGap(SyroChannel *psc, int write_page)
{
	uint8_t ch;
	
	for (ch=0; ch<KORGSYRO_NUM_OF_CHANNEL; ch++) {
		SyroFunc_GenerateSingleCycle(&psc[ch], write_page, 1, false);
	}
}

/*-----------------------------------------------------------------------
	Generate Start Mark
 -----------------------------------------------------------------------*/
void SyroFunc_MakeStartMark(SyroChannel *psc, int write_page)
{
	uint8_t ch;
	
	for (ch=0; ch<KORGSYRO_NUM_OF_CHANNEL; ch++) {
		SyroFunc_GenerateSingleCycle(&psc[ch], write_page, 5, false);
		SyroFunc_SmoothStartMark(&psc[ch], write_page);
	}
}

/*-----------------------------------------------------------------------
	Generate Channel Info
 -----------------------------------------------------------------------*/
void SyroFunc_MakeChannelInfo(SyroChannel *psc, int write_page)
{
	uint8_t ch;
	
	for (ch=0; ch<KORGSYRO_NUM_OF_CHANNEL; ch++) {
		SyroFunc_GenerateSingleCycle(&psc[ch], write_page, ch, true);
	}
}


