package transformers

import (
	"fmt"
	"os"
	"strings"

	"github.com/johnkerl/miller/v6/pkg/cli"
	"github.com/johnkerl/miller/v6/pkg/lib"
	"github.com/johnkerl/miller/v6/pkg/mlrval"
	"github.com/johnkerl/miller/v6/pkg/types"
)

const verbNameUnsparsify = "unsparsify"

var UnsparsifySetup = TransformerSetup{
	Verb:         verbNameUnsparsify,
	UsageFunc:    transformerUnsparsifyUsage,
	ParseCLIFunc: transformerUnsparsifyParseCLI,
	IgnoresInput: false,
}

func transformerUnsparsifyUsage(
	o *os.File,
) {
	fmt.Fprintf(o, "Usage: %s %s [options]\n", "mlr", verbNameUnsparsify)
	fmt.Fprint(o,
		`Prints records with the union of field names over all input records.
For field names absent in a given record but present in others, fills in
a value. This verb retains all input before producing any output.
`)

	fmt.Fprintf(o, "Options:\n")
	fmt.Fprintf(o, "--fill-with {filler string}  What to fill absent fields with. Defaults to\n")
	fmt.Fprintf(o, "                             the empty string.\n")
	fmt.Fprintf(o, "-f {a,b,c} Specify field names to be operated on. Any other fields won't be\n")
	fmt.Fprintf(o, "           modified, and operation will be streaming.\n")
	fmt.Fprintf(o, "-h|--help  Show this message.\n")

	fmt.Fprint(o,
		`Example: if the input is two records, one being 'a=1,b=2' and the other
being 'b=3,c=4', then the output is the two records 'a=1,b=2,c=' and
'a=,b=3,c=4'.
`)
}

func transformerUnsparsifyParseCLI(
	pargi *int,
	argc int,
	args []string,
	_ *cli.TOptions,
	doConstruct bool, // false for first pass of CLI-parse, true for second pass
) (RecordTransformer, error) {

	// Skip the verb name from the current spot in the mlr command line
	argi := *pargi
	verb := args[argi]
	argi++

	fillerString := ""
	var specifiedFieldNames []string = nil

	var err error
	for argi < argc /* variable increment: 1 or 2 depending on flag */ {
		opt := args[argi]
		if !strings.HasPrefix(opt, "-") {
			break // No more flag options to process
		}
		if args[argi] == "--" {
			break // All transformers must do this so main-flags can follow verb-flags
		}
		argi++

		if opt == "-h" || opt == "--help" {
			transformerUnsparsifyUsage(os.Stdout)
			return nil, cli.ErrHelpRequested

		} else if opt == "--fill-with" {
			fillerString, err = cli.VerbGetStringArg(verb, opt, args, &argi, argc)
			if err != nil {
				return nil, err
			}

		} else if opt == "-f" {
			specifiedFieldNames, err = cli.VerbGetStringArrayArg(verb, opt, args, &argi, argc)
			if err != nil {
				return nil, err
			}

		} else {
			return nil, cli.VerbErrorf(verb, "option \"%s\" not recognized", opt)
		}
	}

	*pargi = argi
	if !doConstruct { // All transformers must do this for main command-line parsing
		return nil, nil
	}

	transformer, err := NewTransformerUnsparsify(
		fillerString,
		specifiedFieldNames,
	)
	if err != nil {
		return nil, err
	}

	return transformer, nil
}

type TransformerUnsparsify struct {
	fillerMlrval          *mlrval.Mlrval
	recordsAndContexts    []*types.RecordAndContext
	fieldNamesSeen        *lib.OrderedMap[string]
	recordTransformerFunc RecordTransformerFunc
}

func NewTransformerUnsparsify(
	fillerString string,
	specifiedFieldNames []string,
) (*TransformerUnsparsify, error) {

	fieldNamesSeen := lib.NewOrderedMap[string]()
	for _, specifiedFieldName := range specifiedFieldNames {
		fieldNamesSeen.Put(specifiedFieldName, specifiedFieldName)
	}

	tr := &TransformerUnsparsify{
		fillerMlrval:       mlrval.FromString(fillerString),
		recordsAndContexts: []*types.RecordAndContext{},
		fieldNamesSeen:     fieldNamesSeen,
	}

	if specifiedFieldNames == nil {
		tr.recordTransformerFunc = tr.transformNonStreaming
	} else {
		tr.recordTransformerFunc = tr.transformStreaming
	}

	return tr, nil
}

func (tr *TransformerUnsparsify) Transform(
	inrecAndContext *types.RecordAndContext,
	outputRecordsAndContexts *[]*types.RecordAndContext, // list of *types.RecordAndContext
	inputDownstreamDoneChannel <-chan bool,
	outputDownstreamDoneChannel chan<- bool,
) {
	HandleDefaultDownstreamDone(inputDownstreamDoneChannel, outputDownstreamDoneChannel)
	tr.recordTransformerFunc(inrecAndContext, outputRecordsAndContexts, inputDownstreamDoneChannel, outputDownstreamDoneChannel)
}

func (tr *TransformerUnsparsify) transformNonStreaming(
	inrecAndContext *types.RecordAndContext,
	outputRecordsAndContexts *[]*types.RecordAndContext, // list of *types.RecordAndContext
	inputDownstreamDoneChannel <-chan bool,
	outputDownstreamDoneChannel chan<- bool,
) {
	if !inrecAndContext.EndOfStream {
		inrec := inrecAndContext.Record
		for pe := inrec.Head; pe != nil; pe = pe.Next {
			key := pe.Key
			if !tr.fieldNamesSeen.Has(key) {
				tr.fieldNamesSeen.Put(key, key)
			}
		}
		tr.recordsAndContexts = append(tr.recordsAndContexts, inrecAndContext)
	} else {
		for _, outrecAndContext := range tr.recordsAndContexts {
			outrec := outrecAndContext.Record

			newrec := mlrval.NewMlrmapAsRecord()
			for pe := tr.fieldNamesSeen.Head; pe != nil; pe = pe.Next {
				fieldName := pe.Key
				if !outrec.Has(fieldName) {
					newrec.PutCopy(fieldName, tr.fillerMlrval)
				} else {
					newrec.PutReference(fieldName, outrec.Get(fieldName))
				}
			}

			*outputRecordsAndContexts = append(*outputRecordsAndContexts, types.NewRecordAndContext(newrec, &outrecAndContext.Context))
		}

		*outputRecordsAndContexts = append(*outputRecordsAndContexts, inrecAndContext) // end-of-stream marker
	}
}

func (tr *TransformerUnsparsify) transformStreaming(
	inrecAndContext *types.RecordAndContext,
	outputRecordsAndContexts *[]*types.RecordAndContext, // list of *types.RecordAndContext
	inputDownstreamDoneChannel <-chan bool,
	outputDownstreamDoneChannel chan<- bool,
) {
	if !inrecAndContext.EndOfStream {
		inrec := inrecAndContext.Record

		for pe := tr.fieldNamesSeen.Head; pe != nil; pe = pe.Next {
			if !inrec.Has(pe.Key) {
				inrec.PutCopy(pe.Key, tr.fillerMlrval)
			}
		}

		*outputRecordsAndContexts = append(*outputRecordsAndContexts, inrecAndContext)

	} else {
		*outputRecordsAndContexts = append(*outputRecordsAndContexts, inrecAndContext) // end-of-stream marker
	}
}
