<?php

/*
 * This file is part of the Predis package.
 *
 * (c) 2009-2020 Daniele Alessandri
 * (c) 2021-2026 Till Krüss
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Predis\Command\Redis;

use Predis\Command\PrefixableCommand;

/**
 * @group commands
 * @group realm-key
 */
class EXPIRE_Test extends PredisCommandTestCase
{
    /**
     * {@inheritdoc}
     */
    protected function getExpectedCommand(): string
    {
        return 'Predis\Command\Redis\EXPIRE';
    }

    /**
     * {@inheritdoc}
     */
    protected function getExpectedId(): string
    {
        return 'EXPIRE';
    }

    /**
     * @group disconnected
     */
    public function testFilterArguments(): void
    {
        $arguments = ['key', 'ttl', 'xx'];
        $expected = ['key', 'ttl', 'XX'];

        $command = $this->getCommand();
        $command->setArguments($arguments);

        $this->assertSame($expected, $command->getArguments());
    }

    /**
     * @group disconnected
     */
    public function testParseResponse(): void
    {
        $command = $this->getCommand();

        $this->assertSame(0, $command->parseResponse(0));
        $this->assertSame(1, $command->parseResponse(1));
    }

    /**
     * @group disconnected
     */
    public function testPrefixKeys(): void
    {
        /** @var PrefixableCommand $command */
        $command = $this->getCommand();
        $actualArguments = ['arg1', 'arg2', 'arg3', 'arg4'];
        $prefix = 'prefix:';
        $expectedArguments = ['prefix:arg1', 'arg2', 'arg3', 'arg4'];

        $command->setArguments($actualArguments);
        $command->prefixKeys($prefix);

        $this->assertSame($expectedArguments, $command->getArguments());
    }

    /**
     * @group connected
     */
    public function testReturnsZeroOnNonExistingKeys(): void
    {
        $redis = $this->getClient();

        $this->assertSame(0, $redis->expire('foo', 2));
    }

    /**
     * @group connected
     * @requiresRedisVersion >= 6.0.0
     */
    public function testReturnsZeroOnNonExistingKeysResp3(): void
    {
        $redis = $this->getResp3Client();

        $this->assertSame(0, $redis->expire('foo', 2));
    }

    /**
     * @medium
     * @group connected
     * @group slow
     */
    public function testCanExpireKeys(): void
    {
        $redis = $this->getClient();

        $this->assertEquals('OK', $redis->set('foo', 'bar'));

        $this->assertSame(1, $redis->expire('foo', 1));
        $this->assertSame(1, $redis->ttl('foo'));

        $this->sleep(2.0);
        $this->assertSame(0, $redis->exists('foo'));
    }

    /**
     * @medium
     * @group connected
     * @group slow
     * @dataProvider keysProvider
     * @param  array $firstKeyArguments
     * @param  array $secondKeyArguments
     * @param  array $positivePathArguments
     * @param  array $negativePathArguments
     * @return void
     * @requiresRedisVersion >= 7.0.0
     */
    public function testSetNewExpirationTimeWithExpireOptions(
        array $firstKeyArguments,
        array $secondKeyArguments,
        array $positivePathArguments,
        array $negativePathArguments
    ): void {
        $redis = $this->getClient();

        $redis->set(...$firstKeyArguments);
        $redis->set(...$secondKeyArguments);

        $this->assertSame(1, $redis->expire(...$positivePathArguments));
        $this->assertSame(0, $redis->expire(...$negativePathArguments));
    }

    /**
     * @group connected
     */
    public function testDeletesKeysOnNegativeTTL(): void
    {
        $redis = $this->getClient();

        $this->assertEquals('OK', $redis->set('foo', 'bar'));

        $this->assertSame(1, $redis->expire('foo', -10));
        $this->assertSame(0, $redis->exists('foo'));
    }

    public function keysProvider(): array
    {
        return [
            'only if key has no expiry' => [
                ['noExpiry', 'value'],
                ['withExpiry', 'value', 'EX', 10],
                ['noExpiry', 2, 'NX'],
                ['withExpiry', 2, 'NX'],
            ],
            'only if key has expiry' => [
                ['noExpiry', 'value'],
                ['withExpiry', 'value', 'EX', 10],
                ['withExpiry', 2, 'XX'],
                ['noExpiry', 2, 'XX'],
            ],
            'only if new expiry is greater then current one' => [
                ['newExpiryLower', 'value', 'EX', 1000],
                ['newExpiryGreater', 'value', 'EX', 10],
                ['newExpiryGreater', 100, 'GT'],
                ['newExpiryLower', 10, 'GT'],
            ],
            'only if new expiry is lower then current one' => [
                ['newExpiryLower', 'value', 'EX', 1000],
                ['newExpiryGreater', 'value', 'EX', 10],
                ['newExpiryLower', 10, 'LT'],
                ['newExpiryGreater', 100, 'LT'],
            ],
        ];
    }
}
