/// Flags describing a partition type.
#[derive(Debug, Clone, Copy)]
pub enum PartitionTypeInfoFlags {
    /// No flags set.
    None = 0,
    /// Partition type is used for swap.
    Swap = (1 << 0),
    /// Partition type is used for RAID/LVM or similar.
    Raid = (1 << 1),
    /// Partition type indicates the partition is hidden
    /// (e.g. 'dos' type 0x1b PartitionType::new("Hidden W95 FAT32").
    /// Note that this is not the same as user-toggleable
    /// attributes/flags for a partition.
    Hidden = (1 << 2),
    /// Partition type can only be used when creating a partition
    /// and e.g. should not be selectable in a "change partition type"
    /// user interface (e.g. 'dos' type 0x05, 0x0f and 0x85
    /// for extended partitions).
    CreateOnly = (1 << 3),
    /// Partition type indicates the partition is part of the system / bootloader (e.g. 'dos' types 0xee, 0xff, 'gpt' types for 'EFI System partition' and 'BIOS Boot partition').
    System = (1 << 4),
}

/// Detailed information about a partition type.
///
/// `table_subtype` is used to break the set of partition types for
/// `table_type` into a logical subsets. It is typically only used in
/// user interfaces where the partition type is selected.
#[derive(Debug)]
pub struct PartitionTypeInfo {
    /// A partition table type e.g. `dos` or `gpt`
    pub table_type: &'static str,
    /// A partition table sub-type
    pub table_subtype: &'static str,
    /// A partition type
    pub ty: &'static str,
    /// Name of the partition
    pub(crate) name: &'static str,
    /// Flags describing the partition type
    pub flags: PartitionTypeInfoFlags,
}

impl PartitionTypeInfo {
    const fn new(
        table_type: &'static str,
        table_subtype: &'static str,
        ty: &'static str,
        name: &'static str,
        flags: PartitionTypeInfoFlags,
    ) -> Self {
        //TODO: wrap name with gettext call
        Self {
            table_type,
            table_subtype,
            ty,
            name,
            flags,
        }
    }
}

/// Known [PartitionType]s.
/// see http://en.wikipedia.org/wiki/GUID_Partition_Table
pub(crate) const PARTITION_TYPES: [PartitionTypeInfo; 228] = [
    // Not associated with any OS
    PartitionTypeInfo::new(
        "gpt",
        "generic",
        "024dee41-33e7-11d3-9d69-0008c781f39f",
        "MBR Partition Scheme",
        PartitionTypeInfoFlags::System,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "generic",
        "c12a7328-f81f-11d2-ba4b-00a0c93ec93b",
        "EFI System",
        PartitionTypeInfoFlags::System,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "generic",
        "21686148-6449-6e6f-744e-656564454649",
        "BIOS Boot",
        PartitionTypeInfoFlags::System,
    ),
    // This is also defined in the Apple and Solaris section
    PartitionTypeInfo::new(
        "gpt",
        "generic",
        "6a898cc3-1dd2-11b2-99a6-080020736631",
        "ZFS",
        PartitionTypeInfoFlags::None,
    ),
    // Extended Boot Partition, see http://www.freedesktop.org/wiki/Specifications/BootLoaderSpec/
    PartitionTypeInfo::new(
        "gpt",
        "generic",
        "bc13c2ff-59e6-4262-a352-b275fd6f7172",
        "Extended Boot Partition",
        PartitionTypeInfoFlags::None,
    ),
    // Discoverable Linux Partitions, see http://systemd.io/DISCOVERABLE_PARTITIONS/
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "6523f8ae-3eb1-4e2a-a05a-18b695ae656f",
        "Linux Root Partition (Alpha)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "d27f46ed-2919-4cb8-bd25-9531f3c16534",
        "Linux Root Partition (ARC)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "69dad710-2ce4-4e3c-b16c-21a1d49abed3",
        "Linux Root Partition (32-bit ARM)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "b921b045-1df0-41c3-af44-4c6f280d3fae",
        "Linux Root Partition (64-bit ARM/AArch64)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "993d8d3d-f80e-4225-855a-9daf8ed7ea97",
        "Linux Root Partition (Itanium/IA-64)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "77055800-792c-4f94-b39a-98c91b762bb6",
        "Linux Root Partition (LoongArch 64-bit)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "37c58c8a-d913-4156-a25f-48b1b64e07f0",
        "Linux Root Partition (32-bit MIPS LittleEndian/mipsel)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "700bda43-7a34-4507-b179-eeb93d7a7ca3",
        "Linux Root Partition (64-bit MIPS LittleEndian/mips64el)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "1aacdb3b-5444-4138-bd9e-e5c2239b2346",
        "Linux Root Partition (HPPA/PARISC)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "1de3f1ef-fa98-47b5-8dcd-4a860a654d78",
        "Linux Root Partition (32-bit PowerPC)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "912ade1d-a839-4913-8964-a10eee08fbd2",
        "Linux Root Partition (64-bit PowerPC BigEndian)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "c31c45e6-3f39-412e-80fb-4809c4980599",
        "Linux Root Partition (64-bit PowerPC LittleEndian)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "60d5a7fe-8e7d-435c-b714-3dd8162144e1",
        "Linux Root Partition (RISC-V 32-bit)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "72ec70a6-cf74-40e6-bd49-4bda08e8f224",
        "Linux Root Partition (RISC-V 64-bit)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "08a7acea-624c-4a20-91e8-6e0fa67d23f9",
        "Linux Root Partition (s390)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "5eead9a9-fe09-4a1e-a1d7-520d00531306",
        "Linux Root Partition (s390x)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "c50cdd70-3862-4cc3-90e1-809a8c93ee2c",
        "Linux Root Partition (TILE-Gx)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "44479540-f297-41b2-9af7-d131d5f0458a",
        "Linux Root Partition (x86)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "4f68bce3-e8cd-4db1-96e7-fbcaf984b709",
        "Linux Root Partition (amd64/x86_64)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "e18cf08c-33ec-4c0d-8246-c6c6fb3da024",
        "Linux /usr/ Partition (Alpha)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "7978a683-6316-4922-bbee-38bff5a2fecc",
        "Linux /usr/ Partition (ARC)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "7d0359a3-02b3-4f0a-865c-654403e70625",
        "Linux /usr/ Partition (32-bit ARM)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "b0e01050-ee5f-4390-949a-9101b17104e9",
        "Linux /usr/ Partition (64-bit ARM/AArch64)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "4301d2a6-4e3b-4b2a-bb94-9e0b2c4225ea",
        "Linux /usr/ Partition (Itanium/IA-64)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "e611c702-575c-4cbe-9a46-434fa0bf7e3f",
        "Linux /usr/ Partition (LoongArch 64-bit)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "0f4868e9-9952-4706-979f-3ed3a473e947",
        "Linux /usr/ Partition (32-bit MIPS LittleEndian/mipsel)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "c97c1f32-ba06-40b4-9f22-236061b08aa8",
        "Linux /usr/ Partition (64-bit MIPS LittleEndian/mips64el)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "dc4a4480-6917-4262-a4ec-db9384949f25",
        "Linux /usr/ Partition (HPPA/PARISC)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "7d14fec5-cc71-415d-9d6c-06bf0b3c3eaf",
        "Linux /usr/ Partition (32-bit PowerPC)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "2c9739e2-f068-46b3-9fd0-01c5a9afbcca",
        "Linux /usr/ Partition (64-bit PowerPC BigEndian)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "15bb03af-77e7-4d4a-b12b-c0d084f7491c",
        "Linux /usr/ Partition (64-bit PowerPC LittleEndian)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "b933fb22-5c3f-4f91-af90-e2bb0fa50702",
        "Linux /usr/ Partition (RISC-V 32-bit)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "beaec34b-8442-439b-a40b-984381ed097d",
        "Linux /usr/ Partition (RISC-V 64-bit)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "cd0f869b-d0fb-4ca0-b141-9ea87cc78d66",
        "Linux /usr/ Partition (s390)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "8a4f5770-50aa-4ed3-874a-99b710db6fea",
        "Linux /usr/ Partition (s390x)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "55497029-c7c1-44cc-aa39-815ed1558630",
        "Linux /usr/ Partition (TILE-Gx)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "75250d76-8cc6-458e-bd66-bd47cc81a812",
        "Linux /usr/ Partition (x86)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "8484680c-9521-48c6-9c11-b0720656f69e",
        "Linux /usr/ Partition (amd64/x86_64)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "fc56d9e9-e6e5-4c06-be32-e74407ce09a5",
        "Linux Root Verity Partition (Alpha)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "24b2d975-0f97-4521-afa1-cd531e421b8d",
        "Linux Root Verity Partition (ARC)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "7386cdf2-203c-47a9-a498-f2ecce45a2d6",
        "Linux Root Verity Partition (32-bit ARM)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "df3300ce-d69f-4c92-978c-9bfb0f38d820",
        "Linux Root Verity Partition (64-bit ARM/AArch64)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "86ed10d5-b607-45bb-8957-d350f23d0571",
        "Linux Root Verity Partition (Itanium/IA-64)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "f3393b22-e9af-4613-a948-9d3bfbd0c535",
        "Linux Root Verity Partition (LoongArch 64-bit)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "d7d150d2-2a04-4a33-8f12-16651205ff7b",
        "Linux Root Verity Partition (32-bit MIPS LittleEndian/mipsel)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "16b417f8-3e06-4f57-8dd2-9b5232f41aa6",
        "Linux Root Verity Partition (64-bit MIPS LittleEndian/mips64el)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "d212a430-fbc5-49f9-a983-a7feef2b8d0e",
        "Linux Root Verity Partition (HPPA/PARISC)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "906bd944-4589-4aae-a4e4-dd983917446a",
        "Linux Root Verity Partition (64-bit PowerPC LittleEndian)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "9225a9a3-3c19-4d89-b4f6-eeff88f17631",
        "Linux Root Verity Partition (64-bit PowerPC BigEndian)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "98cfe649-1588-46dc-b2f0-add147424925",
        "Linux Root Verity Partition (32-bit PowerPC)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "ae0253be-1167-4007-ac68-43926c14c5de",
        "Linux Root Verity Partition (RISC-V 32-bit)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "b6ed5582-440b-4209-b8da-5ff7c419ea3d",
        "Linux Root Verity Partition (RISC-V 64-bit)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "7ac63b47-b25c-463b-8df8-b4a94e6c90e1",
        "Linux Root Verity Partition (s390)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "b325bfbe-c7be-4ab8-8357-139e652d2f6b",
        "Linux Root Verity Partition (s390x)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "966061ec-28e4-4b2e-b4a5-1f0a825a1d84",
        "Linux Root Verity Partition (TILE-Gx)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "2c7357ed-ebd2-46d9-aec1-23d437ec2bf5",
        "Linux Root Verity Partition (amd64/x86_64)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "d13c5d3b-b5d1-422a-b29f-9454fdc89d76",
        "Linux Root Verity Partition (x86)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "8cce0d25-c0d0-4a44-bd87-46331bf1df67",
        "Linux /usr/ Verity Partition (Alpha)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "fca0598c-d880-4591-8c16-4eda05c7347c",
        "Linux /usr/ Verity Partition (ARC)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "c215d751-7bcd-4649-be90-6627490a4c05",
        "Linux /usr/ Verity Partition (32-bit ARM)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "6e11a4e7-fbca-4ded-b9e9-e1a512bb664e",
        "Linux /usr/ Verity Partition (64-bit ARM/AArch64)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "6a491e03-3be7-4545-8e38-83320e0ea880",
        "Linux /usr/ Verity Partition (Itanium/IA-64)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "f46b2c26-59ae-48f0-9106-c50ed47f673d",
        "Linux /usr/ Verity Partition (LoongArch 64-bit)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "46b98d8d-b55c-4e8f-aab3-37fca7f80752",
        "Linux /usr/ Verity Partition (32-bit MIPS LittleEndian/mipsel)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "3c3d61fe-b5f3-414d-bb71-8739a694a4ef",
        "Linux /usr/ Verity Partition (64-bit MIPS LittleEndian/mips64el)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "5843d618-ec37-48d7-9f12-cea8e08768b2",
        "Linux /usr/ Verity Partition (HPPA/PARISC)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "ee2b9983-21e8-4153-86d9-b6901a54d1ce",
        "Linux /usr/ Verity Partition (64-bit PowerPC LittleEndian)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "bdb528a5-a259-475f-a87d-da53fa736a07",
        "Linux /usr/ Verity Partition (64-bit PowerPC BigEndian)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "df765d00-270e-49e5-bc75-f47bb2118b09",
        "Linux /usr/ Verity Partition (32-bit PowerPC)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "cb1ee4e3-8cd0-4136-a0a4-aa61a32e8730",
        "Linux /usr/ Verity Partition (RISC-V 32-bit)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "8f1056be-9b05-47c4-81d6-be53128e5b54",
        "Linux /usr/ Verity Partition (RISC-V 64-bit)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "b663c618-e7bc-4d6d-90aa-11b756bb1797",
        "Linux /usr/ Verity Partition (s390)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "31741cc4-1a2a-4111-a581-e00b447d2d06",
        "Linux /usr/ Verity Partition (s390x)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "2fb4bf56-07fa-42da-8132-6b139f2026ae",
        "Linux /usr/ Verity Partition (TILE-Gx)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "77ff5f63-e7b6-4633-acf4-1565b864c0e6",
        "Linux /usr/ Verity Partition (amd64/x86_64)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "8f461b0d-14ee-4e81-9aa9-049b6fb97abd",
        "Linux /usr/ Verity Partition (x86)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "d46495b7-a053-414f-80f7-700c99921ef8",
        "Linux Root Verity Signature Partition (Alpha)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "143a70ba-cbd3-4f06-919f-6c05683a78bc",
        "Linux Root Verity Signature Partition (ARC)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "42b0455f-eb11-491d-98d3-56145ba9d037",
        "Linux Root Verity Signature Partition (32-bit ARM)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "6db69de6-29f4-4758-a7a5-962190f00ce3",
        "Linux Root Verity Signature Partition (64-bit ARM/AArch64)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "e98b36ee-32ba-4882-9b12-0ce14655f46a",
        "Linux Root Verity Signature Partition (Itanium/IA-64)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "5afb67eb-ecc8-4f85-ae8e-ac1e7c50e7d0",
        "Linux Root Verity Signature Partition (LoongArch 64-bit)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "c919cc1f-4456-4eff-918c-f75e94525ca5",
        "Linux Root Verity Signature Partition (32-bit MIPS LittleEndian/mipsel)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "904e58ef-5c65-4a31-9c57-6af5fc7c5de7",
        "Linux Root Verity Signature Partition (64-bit MIPS LittleEndian/mips64el)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "15de6170-65d3-431c-916e-b0dcd8393f25",
        "Linux Root Verity Signature Partition (HPPA/PARISC)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "d4a236e7-e873-4c07-bf1d-bf6cf7f1c3c6",
        "Linux Root Verity Signature Partition (64-bit PowerPC LittleEndian)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "f5e2c20c-45b2-4ffa-bce9-2a60737e1aaf",
        "Linux Root Verity Signature Partition (64-bit PowerPC BigEndian)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "1b31b5aa-add9-463a-b2ed-bd467fc857e7",
        "Linux Root Verity Signature Partition (32-bit PowerPC)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "3a112a75-8729-4380-b4cf-764d79934448",
        "Linux Root Verity Signature Partition (RISC-V 32-bit)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "efe0f087-ea8d-4469-821a-4c2a96a8386a",
        "Linux Root Verity Signature Partition (RISC-V 64-bit)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "3482388e-4254-435a-a241-766a065f9960",
        "Linux Root Verity Signature Partition (s390)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "c80187a5-73a3-491a-901a-017c3fa953e9",
        "Linux Root Verity Signature Partition (s390x)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "b3671439-97b0-4a53-90f7-2d5a8f3ad47b",
        "Linux Root Verity Signature Partition (TILE-Gx)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "41092b05-9fc8-4523-994f-2def0408b176",
        "Linux Root Verity Signature Partition (amd64/x86_64)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "5996fc05-109c-48de-808b-23fa0830b676",
        "Linux Root Verity Signature Partition (x86)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "5c6e1c76-076a-457a-a0fe-f3b4cd21ce6e",
        "Linux /usr/ Verity Signature Partition (Alpha)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "94f9a9a1-9971-427a-a400-50cb297f0f35",
        "Linux /usr/ Verity Signature Partition (ARC)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "d7ff812f-37d1-4902-a810-d76ba57b975a",
        "Linux /usr/ Verity Signature Partition (32-bit ARM)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "c23ce4ff-44bd-4b00-b2d4-b41b3419e02a",
        "Linux /usr/ Verity Signature Partition (64-bit ARM/AArch64)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "8de58bc2-2a43-460d-b14e-a76e4a17b47f",
        "Linux /usr/ Verity Signature Partition (Itanium/IA-64)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "b024f315-d330-444c-8461-44bbde524e99",
        "Linux /usr/ Verity Signature Partition (LoongArch 64-bit)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "3e23ca0b-a4bc-4b4e-8087-5ab6a26aa8a9",
        "Linux /usr/ Verity Signature Partition (32-bit MIPS LittleEndian/mipsel)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "f2c2c7ee-adcc-4351-b5c6-ee9816b66e16",
        "Linux /usr/ Verity Signature Partition (64-bit MIPS LittleEndian/mips64el)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "450dd7d1-3224-45ec-9cf2-a43a346d71ee",
        "Linux /usr/ Verity Signature Partition (HPPA/PARISC)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "c8bfbd1e-268e-4521-8bba-bf314c399557",
        "Linux /usr/ Verity Signature Partition (64-bit PowerPC LittleEndian)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "0b888863-d7f8-4d9e-9766-239fce4d58af",
        "Linux /usr/ Verity Signature Partition (64-bit PowerPC BigEndian)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "7007891d-d371-4a80-86a4-5cb875b9302e",
        "Linux /usr/ Verity Signature Partition (32-bit PowerPC)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "c3836a13-3137-45ba-b583-b16c50fe5eb4",
        "Linux /usr/ Verity Signature Partition (RISC-V 32-bit)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "d2f9000a-7a18-453f-b5cd-4d32f77a7b32",
        "Linux /usr/ Verity Signature Partition (RISC-V 64-bit)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "17440e4f-a8d0-467f-a46e-3912ae6ef2c5",
        "Linux /usr/ Verity Signature Partition (s390)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "3f324816-667b-46ae-86ee-9b0c0c6c11b4",
        "Linux /usr/ Verity Signature Partition (s390x)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "4ede75e2-6ccc-4cc8-b9c7-70334b087510",
        "Linux /usr/ Verity Signature Partition (TILE-Gx)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "e7bb33fb-06cf-4e81-8273-e543b413e2e2",
        "Linux /usr/ Verity Signature Partition (amd64/x86_64)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "974a71c0-de41-43c3-be5d-5c5ccd1ad2c0",
        "Linux /usr/ Verity Signature Partition (x86)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "4d21b016-b534-45c2-a9fb-5c16e091fd2d",
        "Linux Variable Data Partition",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "7ec6f557-3bc5-4aca-b293-16ef5df639d1",
        "Linux Temporary Data Partition",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "773f91ef-66d4-49b5-bd83-d683bf40ad16",
        "Linux Per-user Home Partition",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "933ac7e1-2eb4-4f13-b844-0e14e2aef915",
        "Linux Home Partition",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "3b8f8425-20e0-4f3b-907f-1a25a76f98e8",
        "Linux Server Data Partition",
        PartitionTypeInfoFlags::None,
    ),
    // Linux
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "0657fd6d-a4ab-43c4-84e5-0933c84b4f4f",
        "Linux Swap",
        PartitionTypeInfoFlags::Swap,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "0fc63daf-8483-4772-8e79-3d69d8477de4",
        "Linux Filesystem",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "a19d880f-05fc-4d3b-a006-743f0f84911e",
        "Linux RAID",
        PartitionTypeInfoFlags::Raid,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "e6d6d379-f507-44c2-a23c-238f2a3df928",
        "Linux LVM",
        PartitionTypeInfoFlags::Raid,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "linux",
        "8da63339-0007-60c0-c436-083ac8230908",
        "Linux Reserved",
        PartitionTypeInfoFlags::None,
    ),
    // Microsoft
    PartitionTypeInfo::new(
        "gpt",
        "microsoft",
        "ebd0a0a2-b9e5-4433-87c0-68b6b72699c7",
        "Basic Data",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "microsoft",
        "e3c9e316-0b5c-4db8-817d-f92df00215ae",
        "Microsoft Reserved",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "microsoft",
        "5808c8aa-7e8f-42e0-85d2-e1e90434cfb3",
        "Microsoft LDM metadata",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "microsoft",
        "af9b60a0-1431-4f62-bc68-3311714a69ad",
        "Microsoft LDM data",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "microsoft",
        "de94bba4-06d1-4d40-a16a-bfd50179d6ac",
        "Microsoft Windows Recovery Environment",
        PartitionTypeInfoFlags::None,
    ),
    // Apple OS X
    PartitionTypeInfo::new(
        "gpt",
        "apple",
        "48465300-0000-11aa-aa11-00306543ecac",
        "Apple HFS/HFS+",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "apple",
        "7c3457ef-0000-11aa-aa11-00306543ecac",
        "Apple APFS",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "apple",
        "55465300-0000-11aa-aa11-00306543ecac",
        "Apple UFS",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "apple",
        "6a898cc3-1dd2-11b2-99a6-080020736631",
        "Apple ZFS",
        PartitionTypeInfoFlags::None,
    ), // same as ZFS
    PartitionTypeInfo::new(
        "gpt",
        "apple",
        "52414944-0000-11aa-aa11-00306543ecac",
        "Apple RAID",
        PartitionTypeInfoFlags::Raid,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "apple",
        "52414944-5f4f-11aa-aa11-00306543ecac",
        "Apple RAID (offline)",
        PartitionTypeInfoFlags::Raid,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "apple",
        "426f6f74-0000-11aa-aa11-00306543ecac",
        "Apple Boot",
        PartitionTypeInfoFlags::System,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "apple",
        "4c616265-6c00-11aa-aa11-00306543ecac",
        "Apple Label",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "apple",
        "5265636f-7665-11aa-aa11-00306543ecac",
        "Apple TV Recovery",
        PartitionTypeInfoFlags::System,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "apple",
        "53746f72-6167-11aa-aa11-00306543ecac",
        "Apple Core Storage",
        PartitionTypeInfoFlags::Raid,
    ),
    // HP-UX
    PartitionTypeInfo::new(
        "gpt",
        "other",
        "75894c1e-3aeb-11d3-b7c1-7b03a0000000",
        "HP-UX Data",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "other",
        "e2a1e728-32e3-11d6-a682-7b03a0000000",
        "HP-UX Service",
        PartitionTypeInfoFlags::None,
    ),
    // FreeBSD
    PartitionTypeInfo::new(
        "gpt",
        "other",
        "83bd6b9d-7f41-11dc-be0b-001560b84f0f",
        "FreeBSD Boot",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "other",
        "516e7cb4-6ecf-11d6-8ff8-00022d09712b",
        "FreeBSD Data",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "other",
        "516e7cb5-6ecf-11d6-8ff8-00022d09712b",
        "FreeBSD Swap",
        PartitionTypeInfoFlags::Swap,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "other",
        "516e7cb6-6ecf-11d6-8ff8-00022d09712b",
        "FreeBSD UFS",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "other",
        "516e7cb8-6ecf-11d6-8ff8-00022d09712b",
        "FreeBSD Vinum",
        PartitionTypeInfoFlags::Raid,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "other",
        "516e7cba-6ecf-11d6-8ff8-00022d09712b",
        "FreeBSD ZFS",
        PartitionTypeInfoFlags::None,
    ),
    // Solaris
    PartitionTypeInfo::new(
        "gpt",
        "other",
        "6a82cb45-1dd2-11b2-99a6-080020736631",
        "Solaris Boot",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "other",
        "6a85cf4d-1dd2-11b2-99a6-080020736631",
        "Solaris Root",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "other",
        "6a87c46f-1dd2-11b2-99a6-080020736631",
        "Solaris Swap",
        PartitionTypeInfoFlags::Swap,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "other",
        "6a8b642b-1dd2-11b2-99a6-080020736631",
        "Solaris Backup",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "other",
        "6a898cc3-1dd2-11b2-99a6-080020736631",
        "Solaris /usr",
        PartitionTypeInfoFlags::None,
    ), // same as ZFS
    PartitionTypeInfo::new(
        "gpt",
        "other",
        "6a8ef2e9-1dd2-11b2-99a6-080020736631",
        "Solaris /var",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "other",
        "6a90ba39-1dd2-11b2-99a6-080020736631",
        "Solaris /home",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "other",
        "6a9283a5-1dd2-11b2-99a6-080020736631",
        "Solaris Alternate Sector",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "other",
        "6a945a3b-1dd2-11b2-99a6-080020736631",
        "Solaris Reserved",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "other",
        "6a9630d1-1dd2-11b2-99a6-080020736631",
        "Solaris Reserved (2)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "other",
        "6a980767-1dd2-11b2-99a6-080020736631",
        "Solaris Reserved (3)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "other",
        "6a96237f-1dd2-11b2-99a6-080020736631",
        "Solaris Reserved (4)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "other",
        "6a8d2ac7-1dd2-11b2-99a6-080020736631",
        "Solaris Reserved (5)",
        PartitionTypeInfoFlags::None,
    ),
    // NetBSD
    PartitionTypeInfo::new(
        "gpt",
        "other",
        "49f48d32-b10e-11dc-b99b-0019d1879648",
        "NetBSD Swap",
        PartitionTypeInfoFlags::Swap,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "other",
        "49f48d5a-b10e-11dc-b99b-0019d1879648",
        "NetBSD FFS",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "other",
        "49f48d82-b10e-11dc-b99b-0019d1879648",
        "NetBSD LFS",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "other",
        "49f48daa-b10e-11dc-b99b-0019d1879648",
        "NetBSD RAID",
        PartitionTypeInfoFlags::Raid,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "other",
        "2db519c4-b10f-11dc-b99b-0019d1879648",
        "NetBSD Concatenated",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "other",
        "2db519ec-b10f-11dc-b99b-0019d1879648",
        "NetBSD Encrypted",
        PartitionTypeInfoFlags::None,
    ),
    // VMWare, see http://blogs.vmware.com/vsphere/2011/08/vsphere-50-storage-features-part-7-gpt.html
    PartitionTypeInfo::new(
        "gpt",
        "other",
        "aa31e02a-400f-11db-9590-000c2911d1b8",
        "VMWare VMFS",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "other",
        "9d275380-40ad-11db-bf97-000c2911d1b8",
        "VMWare vmkcore",
        PartitionTypeInfoFlags::None,
    ),
    // ChromeOS, see http://www.chromium.org/chromium-os/chromiumos-design-docs/disk-format
    PartitionTypeInfo::new(
        "gpt",
        "other",
        "cab6e88e-abf3-4102-a07a-d4bb9be3c1d3",
        "ChromeOS Firmware",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "other",
        "fe3a2a5d-4f32-41a7-b725-accc3285a309",
        "ChromeOS Kernel",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "other",
        "3cb8e202-3b7e-47dd-8a3c-7ff2a13cfcec",
        "ChromeOS Root Filesystem",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "gpt",
        "other",
        "2e0a753d-9e48-43b0-8337-b15192cb1b5e",
        "ChromeOS Reserved",
        PartitionTypeInfoFlags::None,
    ),
    // Intel Partition Types
    //     FFS = Fast Flash Standby, aka Intel Rapid start
    //     http://downloadmirror.intel.com/22647/eng/Intel%20Rapid%20Start%20Technology%20Deployment%20Guide%20v1.0.pdf
    PartitionTypeInfo::new(
        "gpt",
        "other",
        "d3bfe2de-3daf-11df-ba40-e3a556d89593",
        "Intel FFS Reserved",
        PartitionTypeInfoFlags::None,
    ),
    // see http://developer.apple.com/documentation/mac/devices/devices-126.html
    //     http://lists.apple.com/archives/Darwin-drivers/2003/May/msg00021.html
    PartitionTypeInfo::new(
        "apm",
        "apple",
        "Apple_Unix_SVR2",
        "Apple UFS",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "apm",
        "apple",
        "Apple_HFS",
        "Apple HFS/HFS",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "apm",
        "apple",
        "Apple_partition_map",
        "Apple Partition Map",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "apm",
        "apple",
        "Apple_Free",
        "Unused",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "apm",
        "apple",
        "Apple_Scratch",
        "Empty",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "apm",
        "apple",
        "Apple_Driver",
        "Driver",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "apm",
        "apple",
        "Apple_Driver43",
        "Driver 4.3",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "apm",
        "apple",
        "Apple_PRODOS",
        "ProDOS file system",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "apm",
        "microsoft",
        "DOS_FAT_12",
        "FAT 12",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "apm",
        "microsoft",
        "DOS_FAT_16",
        "FAT 16",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "apm",
        "microsoft",
        "DOS_FAT_32",
        "FAT 32",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "apm",
        "microsoft",
        "Windows_FAT_16",
        "FAT 16 (Windows)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "apm",
        "microsoft",
        "Windows_FAT_32",
        "FAT 32 (Windows)",
        PartitionTypeInfoFlags::None,
    ),
    // see http://www.win.tue.nl/~aeb/partitions/partition_types-1.html
    PartitionTypeInfo::new(
        "dos",
        "generic",
        "0x05",
        "Extended",
        PartitionTypeInfoFlags::CreateOnly,
    ),
    PartitionTypeInfo::new(
        "dos",
        "generic",
        "0xee",
        "EFI GPT",
        PartitionTypeInfoFlags::System,
    ),
    PartitionTypeInfo::new(
        "dos",
        "generic",
        "0xef",
        "EFI (FAT-12/16/32)",
        PartitionTypeInfoFlags::System,
    ),
    PartitionTypeInfo::new(
        "dos",
        "linux",
        "0x82",
        "Linux swap",
        PartitionTypeInfoFlags::Swap,
    ),
    PartitionTypeInfo::new(
        "dos",
        "linux",
        "0x83",
        "Linux",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "dos",
        "linux",
        "0x85",
        "Linux Extended",
        PartitionTypeInfoFlags::CreateOnly,
    ),
    PartitionTypeInfo::new(
        "dos",
        "linux",
        "0x8e",
        "Linux LVM",
        PartitionTypeInfoFlags::Raid,
    ),
    PartitionTypeInfo::new(
        "dos",
        "linux",
        "0xfd",
        "Linux RAID auto",
        PartitionTypeInfoFlags::Raid,
    ),
    PartitionTypeInfo::new(
        "dos",
        "microsoft",
        "0x01",
        "FAT12",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "dos",
        "microsoft",
        "0x04",
        "FAT16 <32M",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "dos",
        "microsoft",
        "0x06",
        "FAT16",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "dos",
        "microsoft",
        "0x07",
        "NTFS/exFAT/HPFS",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "dos",
        "microsoft",
        "0x0b",
        "W95 FAT32",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "dos",
        "microsoft",
        "0x0c",
        "W95 FAT32 (LBA)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "dos",
        "microsoft",
        "0x0e",
        "W95 FAT16 (LBA)",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "dos",
        "microsoft",
        "0x0f",
        "W95 Ext d (LBA)",
        PartitionTypeInfoFlags::CreateOnly,
    ),
    PartitionTypeInfo::new(
        "dos",
        "microsoft",
        "0x11",
        "Hidden FAT12",
        PartitionTypeInfoFlags::Hidden,
    ),
    PartitionTypeInfo::new(
        "dos",
        "microsoft",
        "0x14",
        "Hidden FAT16 <32M",
        PartitionTypeInfoFlags::Hidden,
    ),
    PartitionTypeInfo::new(
        "dos",
        "microsoft",
        "0x16",
        "Hidden FAT16",
        PartitionTypeInfoFlags::Hidden,
    ),
    PartitionTypeInfo::new(
        "dos",
        "microsoft",
        "0x17",
        "Hidden HPFS/NTFS",
        PartitionTypeInfoFlags::Hidden,
    ),
    PartitionTypeInfo::new(
        "dos",
        "microsoft",
        "0x1b",
        "Hidden W95 FAT32",
        PartitionTypeInfoFlags::Hidden,
    ),
    PartitionTypeInfo::new(
        "dos",
        "microsoft",
        "0x1c",
        "Hidden W95 FAT32 (LBA)",
        PartitionTypeInfoFlags::Hidden,
    ),
    PartitionTypeInfo::new(
        "dos",
        "microsoft",
        "0x1e",
        "Hidden W95 FAT16 (LBA)",
        PartitionTypeInfoFlags::Hidden,
    ),
    PartitionTypeInfo::new("dos", "other", "0x10", "OPUS", PartitionTypeInfoFlags::None),
    PartitionTypeInfo::new(
        "dos",
        "other",
        "0x12",
        "Compaq diagnostics",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "dos",
        "other",
        "0x3c",
        "PartitionMagic",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "dos",
        "other",
        "0x81",
        "Minix",
        PartitionTypeInfoFlags::None,
    ), // cf. http://en.wikipedia.org/wiki/MINIX_file_system
    PartitionTypeInfo::new(
        "dos",
        "other",
        "0x84",
        "Hibernation",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "dos",
        "other",
        "0xa0",
        "Hibernation",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "dos",
        "other",
        "0xa5",
        "FreeBSD",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "dos",
        "other",
        "0xa6",
        "OpenBSD",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "dos",
        "other",
        "0xa8",
        "Mac OS X",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "dos",
        "other",
        "0xaf",
        "Mac OS X",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "dos",
        "other",
        "0xbe",
        "Solaris boot",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "dos",
        "other",
        "0xbf",
        "Solaris",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "dos",
        "other",
        "0xeb",
        "BeOS BFS",
        PartitionTypeInfoFlags::None,
    ),
    PartitionTypeInfo::new(
        "dos",
        "other",
        "0xec",
        "SkyOS SkyFS",
        PartitionTypeInfoFlags::None,
    ),
];
